Attribute VB_Name = "modFTPSync"
Option Explicit

' Sync files to be stored locally on the device
Public Const SYNC_LOCAL_FILE = "local.snc"
Public Const SYNC_REMOTE_FILE = "remote.snc"

' Action constants
Private Const SYNC_ACTION_UPLOAD = 1
Private Const SYNC_ACTION_DOWNLOAD = 2
Private Const SYNC_ACTION_DELETE_LOCAL = 3
Private Const SYNC_ACTION_DELETE_REMOTE = 4

' Priority constants
Public Const SYNC_PRIORITY_LOCAL = 1
Public Const SYNC_PRIORITY_REMOTE = 2
Public Const SYNC_PRIORITY_DATE = 3

' Arrays to store local and remote files
Private sLocalFiles() As String
Private sLocalDates() As Date
Private sRemoteFiles() As String
Private sRemoteDates() As Date

' Arrays to store previous local and remote files
Private sLocalListFiles() As String
Private sLocalListDates() As Date
Private sRemoteListFiles() As String
Private sRemoteListDates() As String

' Arrays to store local actions between local files
Private sLocalActionFiles() As String
Private sLocalActionDates() As Date
Private sLocalActionCommands() As Byte

' Arrays to store remote actions between remote files
Private sRemoteActionFiles() As String
Private sRemoteActionDates() As Date
Private sRemoteActionCommands() As Byte

' Arrays to store the sync actions to be performed
Private sActionListFiles() As String
Private sActionListCommands() As Byte

' Used to identify if there is a sync file to compare with
Private bFirstSync As Boolean


Public Sub SyncDirectories(sServerAddress As String, sServerUsername As String, sServerPassword As String, _
                           sLocalPath As String, sRemotePath As String, _
                           sLocalFileList As String, sRemoteFileList As String, byPriority As Byte)
                                
                                
    Dim lResult As Long
    
    Screen.MousePointer = 11
    
    ' 1) Local the local directory into an array
    SetStatus "Identifying local changes"
    PopulateLocalList sLocalPath
    
    ' 2) Local the local list from file to track changes
    LoadLocalList sLocalFileList
    
    ' 3) Generate an action list for the local directory
    CreateLocalActionList
    
    ' 4) Local the remote directory into an array

    ' Connect to the remote FTP server
    lResult = sFTPOpen(sServerAddress, DEFAULT_FTP_PORT, sServerUsername, sServerPassword, lPassive, FTP_TIMEOUT_SETTING, SapphireFTPLicence)
    If lResult <> FTP_NO_ERROR Then
        SetStatus "Unable to connect to server"
        SetStatus "Error: " & FTPErrorText(lResult)
        Screen.MousePointer = 0
        Exit Sub
    Else
        SetStatus "Connected to remote server"
    End If
    
    SetStatus "Identifying remote changes"
    
    ' Now load the remote diectory into an array
    If PopulateRemoteList(sRemotePath, False) = False Then
        SetStatus "Could not retrieve remote file list"
        sFTPClose (SapphireFTPLicence)
        Screen.MousePointer = 0
        Exit Sub
    End If
    
    ' 5) Load the remote list from file to track changes
    'SetStatus "Loading remote sync list"
    LoadRemoteList sRemoteFileList
    
    ' 6) Generate an action list for the remote directory
    'SetStatus "Determining remote changes"
    CreateRemoteActionList
    
    ' 7) Merge the two action lists to create a sync action list
    'SetStatus "Merging activity lists"
    MergeActionLists byPriority
    
    ' 8) Perform the synchronisation based on the sync action list
    'SetStatus "Running activity list"
    RunActionList sLocalPath
    
    ' 9) Write the local directory back to the local sync file
    'SetStatus "Writing local sync file"
    WriteLocalDirectory sLocalPath, sLocalFileList
    
    ' 10) Write the remote directory back to the remote sync file
    'SetStatus "Writing remote sync file"
    WriteRemoteDirectory sRemotePath, sRemoteFileList
    
    ' Close the FTP session
    lResult = sFTPClose(SapphireFTPLicence)
    If lResult <> FTP_NO_ERROR Then
        SetStatus "Unable to close connection"
        SetStatus "Error: " & FTPErrorText(lResult)
        Screen.MousePointer = 0
        Exit Sub
    Else
        SetStatus "FTP session closed"
    End If
    
    SetStatus "Synchronise completed"
    
    Screen.MousePointer = 0
End Sub

' WriteLocalDirectory stores all the contents of the current local directory
' to file to track changes the next time it syncs
Private Sub WriteLocalDirectory(sLocalPath As String, sLocalFileList As String)
    Dim lCount As Long
    
    ' Create a new local sync file
    On Error Resume Next
    fsControl.Kill sLocalFileList
    
    Err.Clear
    
    fsFile.Open sLocalFileList, fsModeOutput, fsAccessWrite
    If Err.Number <> 0 Then
        SetStatus "Unable to write " & sLocalFileList
        SetStatus "Error: " & Err.Description
        Exit Sub
    End If
    
    On Error GoTo 0
    
    ' Reload the local list
    PopulateLocalList sLocalPath
    
    For lCount = 1 To UBound(sLocalFiles)
        fsFile.LinePrint sLocalFiles(lCount) & "~" & sLocalDates(lCount)
    Next lCount
    
    fsFile.Close
End Sub

' WriteRemoteDirectory stores all the conetns of the current remote directory
' to file to track changes the next time it syncs
Private Sub WriteRemoteDirectory(sRemotePath As String, sRemoteFileList As String)
    Dim lCount As Long
    
    ' Create a new local sync file
    On Error Resume Next
    fsControl.Kill sRemoteFileList
    
    Err.Clear
    
    fsFile.Open sRemoteFileList, fsModeOutput, fsAccessWrite
    If Err.Number <> 0 Then
        SetStatus "Unable to write " & sRemoteFileList
        SetStatus "Error: " & Err.Description
        Exit Sub
    End If
    
    On Error GoTo 0
    
    ' Reload the remote list
    PopulateRemoteList "", True
    
    For lCount = 1 To UBound(sRemoteFiles)
        fsFile.LinePrint sRemoteFiles(lCount) & "~" & sRemoteDates(lCount)
    Next lCount
    
    fsFile.Close
End Sub


' RunActionList processes the action list
Private Sub RunActionList(sLocalPath As String)
    Dim lCount As Long
    
    For lCount = 1 To UBound(sActionListFiles)
        Select Case sActionListCommands(lCount)
            Case SYNC_ACTION_UPLOAD:
                SetStatus "Uploading " & sActionListFiles(lCount)
                SendFile sLocalPath & sActionListFiles(lCount), sActionListFiles(lCount)
            Case SYNC_ACTION_DOWNLOAD:
                SetStatus "Downloading " & sActionListFiles(lCount)
                ReceiveFile sActionListFiles(lCount), sLocalPath & sActionListFiles(lCount)
            Case SYNC_ACTION_DELETE_LOCAL:
                SetStatus "Deleting local file " & sActionListFiles(lCount)
                DeleteLocalFile sLocalPath, sActionListFiles(lCount)
            Case SYNC_ACTION_DELETE_REMOTE:
                SetStatus "Deleting remote file " & sActionListFiles(lCount)
                DeleteRemoteFile sActionListFiles(lCount)
        End Select
    Next lCount
End Sub

' MergeActionLists compares the local and remote action list and determines
' which action has prioity
Private Sub MergeActionLists(byPriority As Byte)
    Dim lCount As Long
    Dim lSearchList As Long
    Dim lSubSearch As Long
    Dim bFound As Boolean
    
    lCount = 1
    
    ' Clear the action lists
    ReDim sActionListFiles(0)
    ReDim sActionListCommands(0)
    
    ' First find all the matching files
    For lSearchList = 1 To UBound(sLocalActionFiles)
        bFound = False
        
        For lSubSearch = 1 To UBound(sRemoteActionFiles)
            If UCase(sLocalActionFiles(lSearchList)) = UCase(sRemoteActionFiles(lSubSearch)) Then
                bFound = True
                
                ' This file has been modified both on the local device and on the server.
                ' Determine which takes priority
                Select Case byPriority
                    Case SYNC_PRIORITY_LOCAL:
                        ' The local action takes priority
                        ReDim Preserve sActionListFiles(lCount)
                        ReDim Preserve sActionListCommands(lCount)
                        
                        sActionListFiles(lCount) = sLocalActionFiles(lSearchList)
                        sActionListCommands(lCount) = sLocalActionCommands(lSearchList)
                        lCount = lCount + 1
                    Case SYNC_PRIORITY_REMOTE:
                        ' The remote action takes priority
                        ReDim Preserve sActionListFiles(lCount)
                        ReDim Preserve sActionListCommands(lCount)
                        
                        sActionListFiles(lCount) = sRemoteActionFiles(lSubSearch)
                        sActionListCommands(lCount) = sRemoteActionCommands(lSubSearch)
                        lCount = lCount + 1
                    Case SYNC_PRIORITY_DATE:
                        ' Compare the file dates and times to determine which takes prioirty
                        
                        ' If local command is to delete and remote to download, then download
                        If (sLocalActionCommands(lSearchList) = SYNC_ACTION_DELETE_REMOTE And _
                           sRemoteActionCommands(lSubSearch) = SYNC_ACTION_DOWNLOAD) Then
                        
                            ReDim Preserve sActionListFiles(lCount)
                            ReDim Preserve sActionListCommands(lCount)
                            
                            sActionListFiles(lCount) = sRemoteActionFiles(lSubSearch)
                            sActionListCommands(lCount) = SYNC_ACTION_DOWNLOAD
                            lCount = lCount + 1
                        End If
                        
                        ' If remote command is to delete and local to upload, then upload
                        If (sLocalActionCommands(lSearchList) = SYNC_ACTION_UPLOAD And _
                            sRemoteActionCommands(lSubSearch) = SYNC_ACTION_DELETE_LOCAL) Then
                        
                            ReDim Preserve sActionListFiles(lCount)
                            ReDim Preserve sActionListCommands(lCount)
                        
                            sActionListFiles(lCount) = sLocalActionFiles(lSearchList)
                            sActionListCommands(lCount) = SYNC_ACTION_UPLOAD
                            lCount = lCount + 1
                        End If
                        
                        ' If local to upload and remote to download then, test dates
                        If (sLocalActionCommands(lSearchList) = SYNC_ACTION_UPLOAD And _
                            sRemoteActionCommands(lSubSearch) = SYNC_ACTION_DOWNLOAD) Then
                        
                            ' Test the dates
                            If DateDiff("s", sLocalActionDates(lSearchList), sRemoteActionDates(lSubSearch)) < 0 Then
                                ' The local file is newer, upload
                                ReDim Preserve sActionListFiles(lCount)
                                ReDim Preserve sActionListCommands(lCount)
                                
                                sActionListFiles(lCount) = sLocalActionFiles(lSearchList)
                                sActionListCommands(lCount) = SYNC_ACTION_UPLOAD
                                lCount = lCount + 1
                            Else
                                ' The remote file is newer, download
                                ReDim Preserve sActionListFiles(lCount)
                                ReDim Preserve sActionListCommands(lCount)
                                
                                sActionListFiles(lCount) = sRemoteActionFiles(lSubSearch)
                                sActionListCommands(lCount) = SYNC_ACTION_DOWNLOAD
                                lCount = lCount + 1
                            End If
                        End If
                End Select
                
                ' Exit the sub loop
                Exit For
            End If
        Next lSubSearch
        
        ' If the local command was not on the server list then set the local command
        ' in the final action list
        If Not bFound Then
            ReDim Preserve sActionListFiles(lCount)
            ReDim Preserve sActionListCommands(lCount)
            
            sActionListFiles(lCount) = sLocalActionFiles(lSearchList)
            sActionListCommands(lCount) = sLocalActionCommands(lSearchList)
            lCount = lCount + 1
        End If
    Next lSearchList

    ' Now search for all the remote actions that are not in the local actions list
    For lSearchList = 1 To UBound(sRemoteActionFiles)
        bFound = False
        
        For lSubSearch = 1 To UBound(sLocalActionFiles)
            If (UCase(sRemoteActionFiles(lSearchList)) = UCase(sLocalActionFiles(lSubSearch))) Then
                ' This has already been processed
                bFound = True
                Exit For
            End If
        Next lSubSearch
        
        If Not bFound Then
            ' This remote action has not been processed so add it to the final list
            ReDim Preserve sActionListFiles(lCount)
            ReDim Preserve sActionListCommands(lCount)
            
            sActionListFiles(lCount) = sRemoteActionFiles(lSearchList)
            sActionListCommands(lCount) = sRemoteActionCommands(lSearchList)
            lCount = lCount + 1
        End If
    Next lSearchList
End Sub

' CreateRemoteActionList compares and remote directory with the
' remote directory from the previous sync and determines what
' has changed
Private Sub CreateRemoteActionList()
    Dim lCount As Long
    Dim lListSearch As Long
    Dim lSubSearch As Long
    Dim bFound As Boolean
    
    lCount = 1
    
    ' Clear the remote action lists
    ReDim sRemoteActionFiles(0)
    ReDim sRemoteActionDates(0)
    ReDim sRemoteActionCommands(0)
    
    ' See if there is no sync file, therefore, everything must set
    ' for uploading
    If bFirstSync Then
        ' Copy each file and set for downloading
        For lCount = 1 To UBound(sRemoteFiles)
            ReDim Preserve sRemoteActionFiles(lCount)
            ReDim Preserve sRemoteActionDates(lCount)
            ReDim Preserve sRemoteActionCommands(lCount)
           
            sRemoteActionFiles(lCount) = sRemoteFiles(lCount)
            sRemoteActionDates(lCount) = sRemoteDates(lCount)
            sRemoteActionCommands(lCount) = SYNC_ACTION_DOWNLOAD
            PrintDebug sRemoteActionFiles(lCount) & " requires downloading"
        Next lCount
    Else
        ' Search through each item in the directory list
        For lListSearch = 1 To UBound(sRemoteFiles)
            bFound = False
            
            ' See if this file exist in the sync file
            For lSubSearch = 1 To UBound(sRemoteListFiles)
                If UCase(sRemoteFiles(lListSearch)) = UCase(sRemoteListFiles(lSubSearch)) Then
                    bFound = True
                    
                    ' See if the file has changes (change in date)
                    If sRemoteDates(lListSearch) <> sRemoteListDates(lSubSearch) Then
                        ' The file has changed, flag it for downloading
                        ReDim Preserve sRemoteActionFiles(lCount)
                        ReDim Preserve sRemoteActionDates(lCount)
                        ReDim Preserve sRemoteActionCommands(lCount)
                        sRemoteActionFiles(lCount) = sRemoteFiles(lListSearch)
                        sRemoteActionDates(lCount) = sRemoteDates(lListSearch)
                        sRemoteActionCommands(lCount) = SYNC_ACTION_DOWNLOAD
                        PrintDebug sRemoteActionFiles(lCount) & " has changed"
                        lCount = lCount + 1
                    End If
                    
                    ' Break out of the sub loop
                    Exit For
                End If
                
            Next lSubSearch
            
            ' If the remote file was not found in the sync list then this indicates
            ' that the file has been added and requires downloading
            If Not bFound Then
                ' The file has been added, flag it for downloading
                ReDim Preserve sRemoteActionFiles(lCount)
                ReDim Preserve sRemoteActionDates(lCount)
                ReDim Preserve sRemoteActionCommands(lCount)
                sRemoteActionFiles(lCount) = sRemoteFiles(lListSearch)
                sRemoteActionDates(lCount) = sRemoteDates(lListSearch)
                sRemoteActionCommands(lCount) = SYNC_ACTION_DOWNLOAD
                PrintDebug sRemoteActionFiles(lCount) & " as been added"
                lCount = lCount + 1
            End If
            
        Next lListSearch
        
        ' Now search the lists and identify any files in the sync file that
        ' are not in the directory list.  This will indicate the file has
        ' been removed.
        For lListSearch = 1 To UBound(sRemoteListFiles)
            bFound = False
            
            For lSubSearch = 1 To UBound(sRemoteFiles)
                If UCase(sRemoteListFiles(lListSearch)) = UCase(sRemoteFiles(lSubSearch)) Then
                    bFound = True
                    Exit For
                End If
            Next lSubSearch
            
            If Not bFound Then
                ' This file has been removed so flag it for deletion on the local device
                ReDim Preserve sRemoteActionFiles(lCount)
                ReDim Preserve sRemoteActionDates(lCount)
                ReDim Preserve sRemoteActionCommands(lCount)
                sRemoteActionFiles(lCount) = sRemoteListFiles(lListSearch)
                sRemoteActionDates(lCount) = sRemoteListDates(lListSearch)
                sRemoteActionCommands(lCount) = SYNC_ACTION_DELETE_LOCAL
                PrintDebug sRemoteActionFiles(lCount) & " has been deleted"
                lCount = lCount + 1
            End If
        Next lListSearch
    End If
    
    ' Display a status if there have been no remote changes
    If UBound(sRemoteActionFiles) = 0 Then
        SetStatus "There have been no remote changes"
    End If
End Sub

' CreateLocalActionList compares and local directory with the
' local directory from the previous sync and determines what
' has changed
Private Sub CreateLocalActionList()
    Dim lCount As Long
    Dim lListSearch As Long
    Dim lSubSearch As Long
    Dim bFound As Boolean
    
    lCount = 1
    
    ' Clear the local action lists
    ReDim sLocalActionFiles(0)
    ReDim sLocalActionDates(0)
    ReDim sLocalActionCommands(0)
    
    ' See if there is no sync file, therefore, everything must set
    ' for uploading
    If bFirstSync Then
        ' Copy each file and set for uploading
        For lCount = 1 To UBound(sLocalFiles)
            ReDim Preserve sLocalActionFiles(lCount)
            ReDim Preserve sLocalActionDates(lCount)
            ReDim Preserve sLocalActionCommands(lCount)
           
            sLocalActionFiles(lCount) = sLocalFiles(lCount)
            sLocalActionDates(lCount) = sLocalDates(lCount)
            sLocalActionCommands(lCount) = SYNC_ACTION_UPLOAD
            PrintDebug sLocalActionFiles(lCount) & " requires uploading"
        Next lCount
    Else
        ' Search through each item in the directory list
        For lListSearch = 1 To UBound(sLocalFiles)
            bFound = False
            
            ' See if this file exist in the sync file
            For lSubSearch = 1 To UBound(sLocalListFiles)
                If UCase(sLocalFiles(lListSearch)) = UCase(sLocalListFiles(lSubSearch)) Then
                    bFound = True
                    
                    ' See if the file has changes (change in date)
                    If sLocalDates(lListSearch) <> sLocalListDates(lSubSearch) Then
                        ' The file has changed, flag it for uploading
                        ReDim Preserve sLocalActionFiles(lCount)
                        ReDim Preserve sLocalActionDates(lCount)
                        ReDim Preserve sLocalActionCommands(lCount)
                        sLocalActionFiles(lCount) = sLocalFiles(lListSearch)
                        sLocalActionDates(lCount) = sLocalDates(lListSearch)
                        sLocalActionCommands(lCount) = SYNC_ACTION_UPLOAD
                        PrintDebug sLocalActionFiles(lCount) & " has changed"
                        lCount = lCount + 1
                    End If
                    
                    ' Break out of the sub loop
                    Exit For
                End If
                
            Next lSubSearch
            
            ' If the local file was not found in the sync list then this indicates
            ' that the file has been added and requires uploading
            If Not bFound Then
                ' The file has been added, flag it for uploading
                ReDim Preserve sLocalActionFiles(lCount)
                ReDim Preserve sLocalActionDates(lCount)
                ReDim Preserve sLocalActionCommands(lCount)
                sLocalActionFiles(lCount) = sLocalFiles(lListSearch)
                sLocalActionDates(lCount) = sLocalDates(lListSearch)
                sLocalActionCommands(lCount) = SYNC_ACTION_UPLOAD
                PrintDebug sLocalActionFiles(lCount) & " was added"
                lCount = lCount + 1
            End If
            
        Next lListSearch
        
        ' Now search the lists and identify any files in the sync file that
        ' are not in the directory list.  This will indicate the file has
        ' been removed
        For lListSearch = 1 To UBound(sLocalListFiles)
            bFound = False
            
            For lSubSearch = 1 To UBound(sLocalFiles)
                If UCase(sLocalListFiles(lListSearch)) = UCase(sLocalFiles(lSubSearch)) Then
                    bFound = True
                    Exit For
                End If
            Next lSubSearch
            
            If Not bFound Then
                ' This file has been removed so flag it for deletion on the server
                ReDim Preserve sLocalActionFiles(lCount)
                ReDim Preserve sLocalActionDates(lCount)
                ReDim Preserve sLocalActionCommands(lCount)
                sLocalActionFiles(lCount) = sLocalListFiles(lListSearch)
                sLocalActionDates(lCount) = sLocalListDates(lListSearch)
                sLocalActionCommands(lCount) = SYNC_ACTION_DELETE_REMOTE
                PrintDebug sLocalActionFiles(lCount) & " was deleted"
                lCount = lCount + 1
            End If
        Next lListSearch
    End If
    
    ' Show a status if there have been no changes
    If UBound(sLocalActionFiles) = 0 Then
        SetStatus "There have been no local changes"
    End If
End Sub

' LoadRemoteList reads the data from a pervious sync into an array
Private Sub LoadRemoteList(sRemoteFile As String)
    Dim sLine As String
    Dim lPos As Long
    Dim lCount As Long
    
    lCount = 1
    ' Clear the remote file list
    ReDim sRemoteListFiles(0)
    ReDim sRemoteListDates(0)
    
    
    If (fsControl.Dir(sRemoteFile) <> "") Then
        bFirstSync = False
    
        fsFile.Open sRemoteFile, fsModeInput, fsAccessRead
    
        ' Read the list data
        While fsFile.EOF = False
            sLine = fsFile.LineInputString()
                 
            ' Extract the filename and the date from this string
            ' seperated by a tilder character (~)
            lPos = InStr(1, sLine, "~")
            If lPos > 1 Then
                ReDim Preserve sRemoteListFiles(lCount)
                ReDim Preserve sRemoteListDates(lCount)
            
                sRemoteListFiles(lCount) = Mid(sLine, 1, lPos - 1)
                sRemoteListDates(lCount) = CDate(Mid(sLine, lPos + 1))
                lCount = lCount + 1
            End If
        Wend
        
        ' Close the file
        fsFile.Close
    Else
        PrintDebug "No remote sync file found"
        bFirstSync = True
    End If
End Sub

' LoadLocalList reads the data from a pervious sync into an array
Private Sub LoadLocalList(sLocalFile As String)
    Dim sLine As String
    Dim lPos As Long
    Dim lCount As Long
    
    lCount = 1
    ' Clear the local file list
    ReDim sLocalListFiles(0)
    ReDim sLocalListDates(0)
    
    
    If (fsControl.Dir(sLocalFile) <> "") Then
        bFirstSync = False
    
        fsFile.Open sLocalFile, fsModeInput, fsAccessRead
    
        ' Read the list data
        While fsFile.EOF = False
            sLine = fsFile.LineInputString()
                 
            ' Extract the filename and the date from this string
            ' seperated by a tilder character (~)
            lPos = InStr(1, sLine, "~")
            If lPos > 1 Then
                ReDim Preserve sLocalListFiles(lCount)
                ReDim Preserve sLocalListDates(lCount)
            
                sLocalListFiles(lCount) = Mid(sLine, 1, lPos - 1)
                sLocalListDates(lCount) = CDate(Mid(sLine, lPos + 1))
                lCount = lCount + 1
            End If
        Wend
        
        ' Close the file
        fsFile.Close
    Else
        PrintDebug "No local sync file found"
        bFirstSync = True
    End If
End Sub

' PopulateLocalList searches for all files and stores them into an array
' with their modified date and time
Private Sub PopulateLocalList(sPath As String)
    Dim sFilename As String
    Dim lCount As Long
    
    ReDim sLocalFiles(0)
    ReDim sLocalDates(0)
    lCount = 1
    
    If Mid(sPath, Len(sPath), 1) <> "\" Then
        sPath = sPath & "\"
    End If
    
    sFilename = fsControl.Dir(sPath & "*.*")
    While sFilename <> ""
        ' Make sure this is not a directory
        If (fsControl.GetAttr(sPath & sFilename) And vbDirectory) <> vbDirectory Then
             ReDim Preserve sLocalFiles(lCount)
             ReDim Preserve sLocalDates(lCount)
             
             sLocalFiles(lCount) = sFilename
             sLocalDates(lCount) = fsControl.FileDateTime(sPath & sFilename)
             lCount = lCount + 1
        End If
        
        sFilename = fsControl.Dir()
    Wend
End Sub

' PopulateRemoteList downloads the list of remote file on the server and
' stores them in a set of arrays
Private Function PopulateRemoteList(sPath As String, bReconnect As Boolean) As Boolean
    Dim sFile As String
    Dim lCount As Long
    Dim lResult As Long
    Dim bySegment As Byte
    Dim sFilename As String
    Dim sFileDate As String
    Dim sAttributes As String
    Dim lStart, lEnd As Long
    
    'SetStatus "Downloading list of remote files"
    
    lCount = 1
    ReDim sRemoteFiles(0)
    ReDim sRemoteDates(0)
    
    If bReconnect Then
        ' Refresh connection to make sure cached items are not used
        If sFTPReconnect(lPassive, SapphireFTPLicence) = 0 Then
            SetStatus "Unable to reconnect"
        End If
    End If
    
    ' If the path is empty then don't change directory
    If sPath <> "" Then
        ' Move to the remote directory
        lResult = sFTPChangeDirectory(sPath, SapphireFTPLicence)
        If lResult <> FTP_NO_ERROR Then
            SetStatus "Unable to locate remote directory"
            SetStatus "Error: " & FTPErrorText(lResult)
                PopulateRemoteList = False
            Exit Function
        End If
    End If
        
    sFile = FTPFindFirstFile("*.*")
    While sFile <> ""
        ' Extract the details out of the file string
        bySegment = 1
        lEnd = InStr(1, sFile, "|")
        While lEnd > 0
            Select Case bySegment
                Case 1:
                    sFilename = Mid(sFile, 1, lEnd - 1)
                Case 5:
                    sFileDate = Mid(sFile, lStart, lEnd - lStart)
                Case 6:
                    sAttributes = Mid(sFile, lStart, lEnd - lStart)
            End Select
            
            lStart = lEnd + 1
            bySegment = bySegment + 1
            lEnd = InStr(lStart, sFile, "|")
        Wend
        
        ' Make sure this file is not a directory
        If (InStr(1, sAttributes, "DIR") = 0) Then
            ReDim Preserve sRemoteFiles(lCount)
            ReDim Preserve sRemoteDates(lCount)
            
            sRemoteFiles(lCount) = sFilename
            sRemoteDates(lCount) = CDate(sFileDate)
            lCount = lCount + 1
        End If
        
        sFile = FTPFindNextFile()
    Wend
    
    PopulateRemoteList = True
End Function

Private Sub SetStatus(sStatus As String)
    ' This routine updates the text box on frmSync to inform the user of any activity
    frmSync.lstStatus.AddItem sStatus
End Sub

Private Sub ProcessFileLists(sLocalPath As String)
    Dim lCount As Long
    Dim bFound As Boolean
    Dim lServerCount As Long
    Dim lResult As Long
    
    ' Make sure the path ends with a trailing slash
    If Mid(sLocalPath, Len(sLocalPath), 1) <> "\" Then
        sLocalPath = sLocalPath & "\"
    End If
    
    ' Process each local file and see if it exists on the server
    For lCount = 1 To UBound(sLocalFiles)
        lServerCount = 1
        bFound = False
        
        While Not bFound And lServerCount <= UBound(sRemoteFiles)
            If UCase(sLocalFiles(lCount)) = UCase(sRemoteFiles(lServerCount)) Then
                bFound = True
            Else
                lServerCount = lServerCount + 1
            End If
        Wend
        
        ' Does the local files exist on the server
        If bFound Then
            ' It exists on the server to see which is the newest
            If DateDiff("s", sLocalDates(lCount), sRemoteDates(lServerCount)) < 0 Then
                ' Send the local file as it is newer
                SetStatus "Local file is newer"
                SendFile sLocalPath & sLocalFiles(lCount), sLocalFiles(lCount)
            Else
                ' Receive the remote file as it is newer
                SetStatus "Server file is newer"
                ReceiveFile sLocalFiles(lCount), sLocalPath & sLocalFiles(lCount)
            End If
        Else
            ' It doesn't exist on the server, so send the local file
            SendFile sLocalPath & sLocalFiles(lCount), sLocalFiles(lCount)
        End If
    Next lCount
    
    ' Now download all server files that were not in the local list
    For lServerCount = 1 To UBound(sRemoteFiles)
        lCount = 1
        bFound = False
    
        While Not bFound And lCount <= UBound(sLocalFiles)
            If UCase(sRemoteFiles(lServerCount)) = UCase(sLocalFiles(lCount)) Then
                bFound = True
            Else
                lCount = lCount + 1
            End If
        Wend
        
        ' See if the server file not exist on the local device
        If Not bFound Then
            ReceiveFile sRemoteFiles(lServerCount), sLocalPath & sRemoteFiles(lServerCount)
        End If
    Next lServerCount
End Sub

Private Sub SendFile(sLocalFile As String, sRemoteFile As String)
    Dim lResult As Long
    
    lResult = sFTPPush(sLocalFile, sRemoteFile, FTP_TRANSFER_BINARY, SapphireFTPLicence)
    If lResult <> FTP_NO_ERROR Then
        SetStatus "Unable to upload " & sRemoteFile
        SetStatus "Error: " & FTPErrorText(lResult)
    Else
        'SetStatus "Sent " & sRemoteFile
    End If
End Sub

Private Sub ReceiveFile(sRemoteFile As String, sLocalFile As String)
    Dim lResult As Long
    
    lResult = sFTPPull(sRemoteFile, sLocalFile, FTP_OVERWRITE, FTP_TRANSFER_BINARY, SapphireFTPLicence)
    If lResult <> FTP_NO_ERROR Then
        SetStatus "Unable to download " & sRemoteFile
        SetStatus "Error: " & FTPErrorText(lResult)
    Else
        'SetStatus "Received " & sRemoteFile
    End If
End Sub

Private Sub DeleteRemoteFile(sRemoteFile As String)
    Dim lResult As Long
    
    lResult = sFTPRemoveFile(sRemoteFile, SapphireFTPLicence)
    If lResult <> FTP_NO_ERROR Then
        SetStatus "Unable to delete remote file " & sRemoteFile
        SetStatus "Error: " & FTPErrorText(lResult)
    Else
        'SetStatus "Deleted remote file " & sRemoteFile
    End If
End Sub

Private Sub DeleteLocalFile(sLocalPath As String, sLocalFile As String)
    On Error Resume Next
    
    Err.Clear
    fsControl.Kill sLocalPath & sLocalFile
    If Err.Number <> 0 Then
        SetStatus "Unable to delete local file " & sLocalFile
        SetStatus "Error: " & Err.Description
    Else
        'SetStatus "Deleted local file " & sLocalFile
    End If
    Err.Clear
    On Error GoTo 0
End Sub

Private Sub PrintDebug(sMessage As String)
    frmSync.lstStatus.AddItem sMessage
End Sub

